#ifndef _PROGRESS_BAR_H_
#define _PROGRESS_BAR_H_


//LIBRARY INFORMATION

#define ProgressBarLibVer1000					0x01000000		//1.0.0.0

#define ProgressBarLibName						"ProgressBarLib"
#define ProgressBarLibType						'libr'
#define ProgressBarLibCreator					'pBar'
#define ProgressBarLibCurVer					ProgressBarLibVer1000


//68K LIBRARY CALL INFORMATION (load as any other library)

#define libTrapProgressBarGetVersion			sysLibTrapCustom+0
#define libTrapProgressBarCreate				sysLibTrapCustom+1
#define libTrapProgressBarDestroy				sysLibTrapCustom+2
#define libTrapProgressBarDraw					sysLibTrapCustom+3
#define libTrapProgressBarSetValues				sysLibTrapCustom+4
#define libTrapProgressBarGetValues				sysLibTrapCustom+5
#define libTrapProgressBarUpdate				sysLibTrapCustom+6
#define libTrapProgressBarErase					sysLibTrapCustom+7


//ARM LIBRARY CALL INFORMATION (version=1, number of entries=8)
//							 in ARM library mode all function prototypes are same, except they lack the "libRef" parameter

#define libArmEntryNumProgressBarGetVersion		0
#define libArmEntryNumProgressBarCreate			1
#define libArmEntryNumProgressBarDestroy		2
#define libArmEntryNumProgressBarDraw			3
#define libArmEntryNumProgressBarSetValues		4
#define libArmEntryNumProgressBarGetValues		5
#define libArmEntryNumProgressBarUpdate			6
#define libArmEntryNumProgressBarErase			7


//DEFITINIONS

typedef UInt32	progressBarPtr;					//this pointer will be passed to all ProgressBar functions

typedef Boolean (*strFunc)(progressBarPtr p,char* buf,UInt32 min,UInt32 max,UInt32 val,UInt32 percent,UInt32 percentDec);	
										//function to create custom text, return true to continue drawing bar...

#define strStdFuncNone		((strFunc)0)				//pass this as callback to have the bar display no text
#define strStdFuncPercent	((strFunc)1)				//pass this as callback to have the bar display percents (eg: "55.9 %")
#define strStdFuncVal		((strFunc)2)				//pass this as callback to have the bar current value as text (eg: "436")
#define strStdFuncValMax	((strFunc)3)				//pass this as callback to have the bar current and max values as text (eg: "77/100)


#define pBarStyleAttrGraphical			2			//graphical: bars use bitmaps to display the bar itself
#define pBarStyleAttrCool			1			//cool:  insted of different colors for text in filled and not filled part of bar, text is drawn
									//       with an outline. (inner color "f", outer color is "e", where "e" and "f" are parameters
									//       to ProgressBarCreate 


#define pBarStyleGraphicsAndText		0 | pBarStyleAttrGraphical
#define pBarStyleGraphicsAndCoolText		0 | pBarStyleAttrGraphical | pBarStyleAttrCool
#define pBarStyleColorsAndText			0
#define pBarStyleColorsAndCoolText		0 | pBarStyleAttrCool

//FUNCTIONS

/**
 *  @brief Retrieve the version of the library
 *
 *  @param libRef:	IN:  library reference number [from SysLibLoad() or SysLibFind()]
 *  @retval UInt32 Version (highest byte is major version, next is minor, next is revision, and lowest byte is bugfix)
 **/
UInt32 ProgressBarGetVersion(UInt16 libRef)
	SYS_TRAP(libTrapProgressBarGetVersion);


/**
 *  @brief Create a progress bar (this will not actually draw it)
 *
 *  @param libRef:	IN:  library reference number [from SysLibLoad() or SysLibFind()]
 *  @param x:	IN:  X-coordinate of the top left corner of the progress bar
 *  @param y:	IN:  Y-coordinate of the top left corner of the progress bar
 *  @param w:	IN:  Width of the progress bar
 *  @param h:	IN:  Height of the progress bar
 *  @param style:	IN:  pass one of the "pBarStyle*" constants
 *  @param striFunc:	IN:  a callback to form the string to be drawn on the bar: pass a callback OR one of the "strStdFunc*" values to have one of the standard strings shown
 *  @param textEmpty:	IN:  Pointer to the color used to draw the text over the non-filled area of the bar, the library keeps an internal copy of this paramater, so the pointer need not be valid after this call
 *  @param textFull:	IN:  Pointer to the color used to draw the text over the filled area of the bar, the library keeps an internal copy of this paramater, so the pointer need not be valid after this call
 *  @param e:	IN:  If this is a "graphical" bar this is a UInt16 representing the resoruce ID of the bitmap to show the unfilled area of the bar, cast to a void*; if bar is not graphical, this is a pointer to a RGBColorType representing the color of the non-filled area of the bar, the library keeps an internal copy of this paramater, so the pointer need not be valid after this call
 *  @param f:	IN:  If this is a "graphical" bar this is a UInt16 representing the resoruce ID of the bitmap to show the filled area of the bar, cast to a void*; if bar is not graphical, this is a pointer to a RGBColorType representing the color of the non0filled area of the bar, the library keeps an internal copy of this paramater, so the pointer need not be valid after this call
 *  @param f:	IN:  If this is a "graphical" bar this value is not used; if bar is not graphical, this is a pointer to a RGBColorType representing the color of the frame around the bar
 *  @retval progressBarPtr Non-zero progressBarPtr is successful, NULL otherwise
 **/
progressBarPtr ProgressBarCreate(UInt16 libRef,UInt16 x,UInt16 y,UInt16 w,UInt16 h,UInt8 style,strFunc striFunc,
 					RGBColorType* textEmpty,RGBColorType* textFull,void* e,void* f,void* b)
	SYS_TRAP(libTrapProgressBarCreate);


/**
 *  @brief Destroy a progress bar
 *
 *  @param libRef:	IN:  library reference number [from SysLibLoad() or SysLibFind()]
 *  @param p:	IN:  progressBarPtr of the progress bar
 *  @retval void
 **/
void ProgressBarDestroy(UInt16 libRef,progressBarPtr p)
	SYS_TRAP(libTrapProgressBarDestroy);


/**
 *  @brief Draw a progress bar
 *
 *  @param libRef:	IN:  library reference number [from SysLibLoad() or SysLibFind()]
 *  @param p:	IN:  progressBarPtr of the progress bar
 *  @retval void
 **/
void ProgressBarDraw(UInt16 libRef,progressBarPtr p)
	SYS_TRAP(libTrapProgressBarDraw);


/**
 *  @brief Set a progressbar's minimum, maximum and current values
 *
 *  @param libRef:	IN:  library reference number [from SysLibLoad() or SysLibFind()]
 *  @param p:	IN:  progressBarPtr of the progress bar
 *  @param min:	IN:  Minimum value of the bar (usually 0)
 *  @param max:	IN:  Maximum value of the bar (eg: total size of data to be received)
 *  @param val:	IN:  Current value of the bar (eg: size of data already received)
 *  @retval UInt8 0 on success, 1 if min>=max, 2 if val<min OR val>max
 **/
UInt8 ProgressBarSetValues(UInt16 libRef,progressBarPtr p,UInt32 min,UInt32 max,UInt32 val)
	SYS_TRAP(libTrapProgressBarSetValues);


/**
 *  @brief Get a progressbar's minimum, maximum or current values
 *
 *  @param libRef:	IN:  library reference number [from SysLibLoad() or SysLibFind()]
 *  @param p:	IN:  progressBarPtr of the progress bar
 *  @param min:	IN:  Pointer to a UInt32 where to store the minimum value of the bar, or NULL of you do not want this information
 *  @param max:	IN:  Pointer to a UInt32 where to store the maximum value of the bar, or NULL of you do not want this information
 *  @param val:	IN:  Pointer to a UInt32 where to store the current value of the bar, or NULL of you do not want this information
 *  @retval void
 **/
void ProgressBarGetValues(UInt16 libRef,progressBarPtr p,UInt32* min,UInt32* max,UInt32* val)
	SYS_TRAP(libTrapProgressBarGetValues);

/**
 *  @brief Update a progessbar's state. [this is the same as calling ProgressBarGetValues(), modifying val, and calling ProgressBarSetValues()]
 *
 *  @param libRef:	IN:  library reference number [from SysLibLoad() or SysLibFind()]
 *  @param p:	IN:  progressBarPtr of the progress bar
 *  @param val:	IN:  New value of the bar
 *  @param redraw:	IN:  Set to redraw the bar immediately
 *  @retval UInt8 0 on success, 1 if val<bar.min OR val>bar.max
 **/
UInt8 ProgressBarUpdate(UInt16 libRef,progressBarPtr p,UInt32 val,Boolean redraw)
	SYS_TRAP(libTrapProgressBarUpdate);


/**
 *  @brief Erase a progress bar
 *
 *  @param libRef:	IN:  library reference number [from SysLibLoad() or SysLibFind()]
 *  @param p:	IN:  progressBarPtr of the progress bar
 *  @retval void none
 **/
void ProgressBarErase(UInt16 libRef,progressBarPtr p)
	SYS_TRAP(libTrapProgressBarErase);






#endif //_PROGRESS_BAR_H_